/*****************************************************************************
 * ugBASIC - an isomorphic BASIC language compiler for retrocomputers        *
 *****************************************************************************
 * Copyright 2021-2022 Marco Spedaletti (asimov@mclink.it)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *----------------------------------------------------------------------------
 * Concesso in licenza secondo i termini della Licenza Apache, versione 2.0
 * (la "Licenza"); è proibito usare questo file se non in conformità alla
 * Licenza. Una copia della Licenza è disponibile all'indirizzo:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Se non richiesto dalla legislazione vigente o concordato per iscritto,
 * il software distribuito nei termini della Licenza è distribuito
 * "COSÌ COM'È", SENZA GARANZIE O CONDIZIONI DI ALCUN TIPO, esplicite o
 * implicite. Consultare la Licenza per il testo specifico che regola le
 * autorizzazioni e le limitazioni previste dalla medesima.
 ****************************************************************************/

/****************************************************************************
 * INCLUDE SECTION 
 ****************************************************************************/

#include "../../ugbc.h"
#define STB_IMAGE_IMPLEMENTATION
#include "../../stb_image.h"
#include "../../libs/msc1.h"

/****************************************************************************
 * CODE SECTION 
 ****************************************************************************/

/**
 * @brief Emit code for <strong>LOAD IMAGE(...)</strong>
 * 
 * @param _environment Current calling environment
 * @param _filename Filename to read into buffer
 * @param _mode Mode to use to convert data
 */
/* <usermanual>
@keyword LOAD IMAGE

@english
The ''LOAD IMAGE'' command allows you to load an image and to convert it into
an ''IMAGE''. The second parameter is the mode to use to convert
the given data (by default, it is equal to current mode)

The command support a set of modern image format, like:

  * JPEG baseline & progressive
  * PNG 1/2/4/8/16-bit-per-channel
  * TGA
  * BMP (non-1bpp, non-RLE)
  * PSD (composited view only, no extra channels, 8/16 bit-per-channel)
  * GIF
  * HDR (radiance rgbE format)
  * PIC (Softimage PIC)
  * PNM (PPM and PGM binary only)

The image will be converted into a way that can be efficiently drawn
on the screen. It could be converted into indexed palette, and can be
rescaled as well.

Since it is possible to load only one file of the same type at a time, it is necessary 
to be able to indicate an "alias" with which to exceed this limit. In this regard, there is also
the ''AS'' syntax, which allows you to load the same file several times but with different names.

@italian
Il comando ''LOAD IMAGE'' permette di caricare un file immagine, e di convertirlo
in una ''IMAGE''. Il secondo parametro è la modalità grafica da usare
per convertire il dato (per default, è il modo corrente).

Il comando supporta una serie di formati moderni:

  * JPEG baseline & progressive
  * PNG 1/2/4/8/16-bit-per-canale
  * TGA
  * BMP (non-1bpp, non-RLE)
  * PSD (vista composita, nessun canale extra, 8/16 bit-per-canale)
  * GIF
  * HDR (formato radiance rgbE)
  * PIC (Softimage PIC)
  * PNM (solo formato binario PPM e PGM)

L'immagine verrà convertita in un modo che possa essere disegnata in modo efficiente
sullo schermo. Potrebbe essere convertita in una tavolozza indicizzata, e potrebbe essere
anche ridimensionata.

Dal momento in cui è possibile caricare un solo file dello stesso tipo alla volta, 
è necessario poter indicare un "alias" con cui superare questo limite. A tal riguardo 
esiste anche la sintassi ''AS'', che permette di caricare più volte lo stesso file 
ma con nomi diversi.

@syntax = LOAD IMAGE([filename]{,[mode]})
@syntax = LOAD IMAGE([filename] AS [alias]{,[mode]})

@example starship = LOAD IMAGE("starship.png")
@example starship2 = LOAD IMAGE("starship.png" AS "starship2")
@example alienAt11 = LOAD IMAGE("alien.jpg",11)
@example alien2 = LOAD IMAGE("alien.jpg" AS "alien2",11)

@usedInExample image_loading_01.bas

@target all
</usermanual> */
Variable * image_load( Environment * _environment, char * _filename, char * _alias, int _mode, int _flags, int _transparent_color, int _background_color, int _bank_expansion ) {

    if ( _environment->tenLinerRulesEnforced ) {
        CRITICAL_10_LINE_RULES_ENFORCED( "LOAD IMAGE");
    }
    
    LoadedFile * first = _environment->loadedFiles;
    char *lookfor = _filename;
    if ( _alias ) {
        lookfor = _alias;
    }
    while( first ) {
        if ( strcmp(lookfor, first->fileName ) == 0 ) {
            return first->variable;
        }
        first = first->next;
    }

    int width = 0;
    int height = 0;
    int depth = 3;

    char * lookedFilename = image_load_asserts( _environment, _filename );

    unsigned char* source = stbi_load(lookedFilename, &width, &height, &depth, 0);

    if ( !source ) {
        CRITICAL_IMAGE_LOAD_UNKNOWN_FORMAT( _filename );
    }

    if( _flags & FLAG_FLIP_X ) {
        source = image_flip_x( _environment, source, width, height );
    }
    if( _flags & FLAG_FLIP_Y ) {
        source = image_flip_y( _environment, source, width, height );
    }

    if ( _transparent_color != -1 ) {
        _flags |= FLAG_TRANSPARENCY;
    }
    
    Variable * result = image_converter( _environment, source, width, height, 0, 0, 0, 0, _mode, _transparent_color, _flags );
    
    result->originalBitmap = source;
    result->originalWidth = width;
    result->originalHeight = height;
    if ( _bank_expansion && _environment->expansionBanks ) {

        // Try to compress the result of image conversion.
        // This means that the buffer will be compressed using MSC1
        // algorithm, up to 32 frequent sequences. The original size of
        // the buffer will be considered as "uncompressed" size.
        MSC1Compressor * compressor = msc1_create( 32 );
        result->uncompressedSize = result->size;
        MemoryBlock * output = msc1_compress( compressor, result->valueBuffer, result->uncompressedSize, &result->size );

        // printf("\n0000: " );
        // for( int k=0; k<result->size; ++k ) {
        //     printf( "%2.2x ", output[k] );
        //     if ( ((k+1) % 8 == 0 ) ) {
        //         printf("\n%4.4x: ", (k) );
        //     };
        // }

        int temporary;
        MemoryBlock * outputCheck = msc1_uncompress( compressor, output, result->size, &temporary );
        // printf("\n0000: " );
        // for( int k=0; k<result->uncompressedSize; ++k ) {
        //     printf( "%2.2x ", result->valueBuffer[k] );
        //     if ( ((k+1) % 8 == 0 ) ) {
        //         printf("\n%4.4x: ", (k) );
        //     };
        // }

        // printf("\n0000: " );
        // for( int k=0; k<result->uncompressedSize; ++k ) {
        //     if ( outputCheck[k] == result->valueBuffer[k] ) {
        //         printf( "        " );
        //     } else {
        //         printf( "%2.2x [%2.2x] ", outputCheck[k], result->valueBuffer[k] );
        //     }
        //     if ( ((k+1) % 8 == 0 ) ) {
        //         printf("\n%4.4x: ", (k) );
        //     };
        // }
        if ( memcmp( outputCheck, result->valueBuffer, result->uncompressedSize ) != 0 ) {
            CRITICAL("Compression failed");
        }
        msc1_free( compressor );

        printf( "%s: %d bytes -> %d bytes\n", _filename, result->uncompressedSize, result->size );
        // If the compressed memory is greater than the original
        // size, we discard the compression and we will continue as
        // usual.
        if ( result->uncompressedSize < result->size ) {
            result->size = result->uncompressedSize;
            result->uncompressedSize = 0;
            free( output );
        } 
        // Otherwise, we can safely replace the original data
        // buffer with the compressed one.
        else {
            free( result->valueBuffer );
            result->valueBuffer = output;
        }

        Bank * bank = _environment->expansionBanks;

        while( bank ) {
            if ( bank->remains > result->size ) {
                break;
            }
            bank = bank->next;
        } 

        if ( ! bank ) {
            CRITICAL_EXPANSION_OUT_OF_MEMORY_LOADING( _filename );
        }

        result->bankAssigned = bank->id;
        result->absoluteAddress = bank->address;
        result->residentAssigned = _bank_expansion;
        result->variableUniqueId = UNIQUE_RESOURCE_ID;
        memcpy( &bank->data[bank->address], result->valueBuffer, result->size );

        bank->address += result->size;
        bank->remains -= result->size;

        // Now we must calculate the effective size occupied by
        // memory block, when it will be uncompressed. It is needed
        // to have enough memory into the resident part of the
        // memory. If uncompressed size is zero, it means that
        // the memory block is not compressed -- so we can use the
        // size as well.
        int realSize = result->uncompressedSize;
        if ( realSize == 0 ) realSize = result->size;

        if ( _environment->maxExpansionBankSize[_bank_expansion] < realSize ) {
            _environment->maxExpansionBankSize[_bank_expansion] = realSize;
        }

    } else if ( _flags & FLAG_COMPRESSED ) {

        // Try to compress the result of image conversion.
        // This means that the buffer will be compressed using MSC1
        // algorithm, up to 32 frequent sequences. The original size of
        // the buffer will be considered as "uncompressed" size.
        MSC1Compressor * compressor = msc1_create( 32 );
        result->uncompressedSize = result->size;
        MemoryBlock * output = msc1_compress( compressor, result->valueBuffer, result->uncompressedSize, &result->size );

        int temporary;
        MemoryBlock * outputCheck = msc1_uncompress( compressor, output, result->size, &temporary );
        // printf("\n0000: " );
        // for( int k=0; k<result->uncompressedSize; ++k ) {
        //     printf( "%2.2x ", result->valueBuffer[k] );
        //     if ( ((k+1) % 8 == 0 ) ) {
        //         printf("\n%4.4x: ", (k) );
        //     };
        // }
        // printf("\n0000: " );
        // for( int k=0; k<result->uncompressedSize; ++k ) {
        //     if ( outputCheck[k] == result->valueBuffer[k] ) {
        //         printf( "        " );
        //     } else {
        //         printf( "%2.2x [%2.2x] ", outputCheck[k], result->valueBuffer[k] );
        //     }
        //     if ( ((k+1) % 8 == 0 ) ) {
        //         printf("\n%4.4x: ", (k) );
        //     };
        // }
        if ( memcmp( outputCheck, result->valueBuffer, result->uncompressedSize ) != 0 ) {
            CRITICAL("Compression failed");
        }
        msc1_free( compressor );
        printf( "%s: %d bytes -> %d bytes\n", _filename, result->uncompressedSize, result->size );
        // If the compressed memory is greater than the original
        // size, we discard the compression and we will continue as
        // usual.
        if ( result->uncompressedSize < result->size ) {
            result->size = result->uncompressedSize;
            result->uncompressedSize = 0;
            free( output );
        } 
        // Otherwise, we can safely replace the original data
        // buffer with the compressed one.
        else {
            free( result->valueBuffer );
            result->valueBuffer = output;
        }
        result->residentAssigned = 1;
        _environment->maxExpansionBankSize[1] = BANK_SIZE;

    }

    // stbi_image_free(source);

    LoadedFile * loaded = malloc( sizeof( LoadedFile ) );
    loaded->next = first;
    loaded->variable = result;
    loaded->fileName = lookfor;
    _environment->loadedFiles = loaded;

    if ( _alias ) {
        const_define_numeric( _environment, _alias, UNIQUE_RESOURCE_ID );
    }
    
    return result;

}
